﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CashierCrisis.Graphs
{
    public class SparseGraph<N, E> 
        where N : GraphNode
        where E : GraphEdge
    {
        private List<N> _nodes = new List<N>();
        private List<List<E>> _edges = new List<List<E>>();
        private bool _digraph;
        private int _nextNodeIndex;


        public List<N> Nodes
        {
            get { return _nodes; }
        }

        public bool IsDiGraph
        {
            get { return _digraph; }
        }

        public bool Empty
        {
            get { return _nodes.Count == 0; }
        }

        public int NodeCount
        {
            get { return _nodes.Count; }
        }

        public bool ContainsNode(int index)
        {
            if (index > NodeCount || index == GraphNode.InvalidNodeIndex)
                return false;
            return true;
        }

        public SparseGraph(bool directed)
        {
            _nextNodeIndex = 0;
            _digraph = directed;
        }



        public bool IsUniqueEdge(int to, int from)
        {
            foreach (E edge in Edges())
            {
                if (edge.To == to && edge.From == from)
                {
                    return false;
                }
               
            }
            return true;
        }

        public N GetNode(int index)
        {
            return _nodes[index];
        }

        public E GetEdge(int to, int from)
        {
            return _edges[from][to];
        }

        public void SetEdgeCost(int to, int from, double cost)
        {
            _edges[from][to].Cost = cost;
        }

        public IEnumerable<E> Edges()
        {
            foreach (List<E> list in _edges)
            {
                foreach(E edge in list)
                {
                    yield return edge; 
                }
            }
        }

        public IEnumerable<E> Edges(int nodeIndex)
        {
            foreach (List<E> edgeList in _edges)
            {
                foreach (E edge in edgeList)
                {
                    if (edge.From == nodeIndex || edge.To == nodeIndex)
                    {
                        yield return edge;
                    }
                }
            }
            
        }

        


        public int NextFreeNodeIndex
        {
            get
            {
                return _nextNodeIndex;
            }
        }

        public int AddNode(N node)
        {
            if (node.Index < NodeCount)
            {
                _nodes[node.Index] = node;

                return _nextNodeIndex;
            }
            else
            {
                _nodes.Add(node);
                _edges.Add(new List<E>());
            }

            return _nextNodeIndex++;
        }

        public void RemoveNode(int node)
        {
            _nodes[node].Index = GraphNode.InvalidNodeIndex;
        }

        public void Clear()
        {
            _nodes.Clear();
            _edges.Clear();
        }

        public void AddEdge(E edge)
        {
            if (_nodes[edge.To].Index != GraphNode.InvalidNodeIndex &&
                _nodes[edge.From].Index != GraphNode.InvalidNodeIndex)
            {
                if (IsUniqueEdge(edge.To, edge.From))
                {
                    _edges[edge.From].Add(edge);
                }
            }

            if (!_digraph)
            {
                if(IsUniqueEdge(edge.To, edge.From))
                {
                    GraphEdge temp = new GraphEdge();

                    temp.To = edge.From;
                    temp.From = edge.To;


                    E newEdge = temp as E;

                    _edges[edge.To].Add(newEdge);
                }
            }
        }
    }
}
